{ This script is frequency-domain induction in a concentric 2D Earth with spatially
uniform, purely real, periodic, external B-field in the z-direction only, using 
the vector potential.  The system is a cross-section of Earth at 0 and 180 deg 
longitude, viewed from the side.  x is to the right, y is into the page, z is up.
Note on units: length units are Mm = 1e6m.  Thus we must convert permeability 
(1H/m =1 kg*m/(s*A)^ 2), and conductivity (1 S/m = 1s^ 3*A^ 2/(kg*m^ 3)). }

TITLE '2D induction from the side using vector potential in the frequency domain'

SELECT
	ngrid = 2				{ for large cells at space boundary }
	stages = 35				{ number of frequencies to simulate }
	threads = 4				{ number of CPU cores to use }
	prefer_stability			
	errlim = 1e-3				{ error tolerance }
	gridlimit = 20

COORDINATES
	cartesian2 ('X','Z')

VARIABLES
 	A_syr					{ Re(z-comp) of secondary vector pot'l}
	A_syi					{ Im(z-comp) of secondary vector pot'l}

DEFINITIONS
	dens = 25				{ for controlling mesh density }
	M = 1e6					{ 1 million }
	mu0 = 4*pi*1e-7/M			{ permeability of free space (see note on units) }			
	mrad = 6.371				{ radius of mantle in Mm }
	srad = 10*mrad				{ full radius of modeled space }
	crad = 3.5				{ radius of outer core }
	tzb = mrad-.8				{ radii of conductivity interfaces }
	tzt = mrad-.4
	zm = 1.5*mrad				{ for zooming plots } 	
	sat_alt = .4			
	sat = mrad + sat_alt
	B0 = 100e-9				{ amplitude of primary field }	
	logperiod = -4/24*(stage)+8+4/24	{ goes from a few minutes to ~3 years in 35 steps }
	period = 10^logperiod			{ primary field period in seconds }
	f = 1/period
	w = 2*pi*f				{ angular frequency of primary field }
	sigma = 0*M^3				{ space conductivity }
	sigma_core = 5e5*M^3			{ core conductivity }
	sigma_lm = 1*M^3			{ lower mantle conductivity }
	sigma_tz = .1*M^3			{ transition zone conductivity }
	sigma_um = .01*M^3			{ upper mantle conductivity }
	tau = 9000*M^2				{ ocean conductance }

	{ create primary (external) potential and field }
	A_pyr = B0/2*x				{ Re(y-comp) of primary vector potential }
	A_pyi = 0
	B_pxr = -dz(A_pyr)
	B_pzr = dx(A_pyr)+A_pyr/x
	B_pr = vector(B_pxr, B_pzr)		{ Re(primary mag field) }	

	{ create secondary (internal) field }
	B_sxr = -dz(A_syr)			{ Re(y-comp) of secondary vector pot'l }
	B_szr = dx(A_syr)+A_syr/x
	B_sr = vector(B_sxr,B_szr)		{ Re(secondary mag field) }
	B_sxi = -dz(A_syi)
	B_szi = dx(A_syi)+A_syi/x
	B_si = vector(B_sxi,B_szi)		{ Im(secondary mag field) }

	{ create total potential and field }
	A_tyr = A_pyr + A_syr			{ Re(y-comp) of total vector pot'l }
	B_txr = B_pxr + B_sxr
	B_tzr = B_pzr + B_szr
	B_tr = B_pr + B_sr			{ Re(total mag field) }
	A_tyi = A_syi				{ Im(y-comp) of total vector pot'l }
	B_txi = B_sxi	
	B_tzi = B_szi	
	B_ti = B_si				{ Im(total mag field) }
	E_r = w*A_tyi				{ Re(induced electric field) }
	E_i = -w*A_tyr				{ Im(induced electric field) }

EQUATIONS
	A_syr:	del2(A_syr) + dx(A_syr)/x - A_syr/x^2 = -w*mu0*sigma*A_tyi
	A_syi:	del2(A_syi)  + dx(A_syi)/x - A_syi/x^2 = w*mu0*sigma*A_tyr
	
RESOLVE			{ Include mag fields in error estimate for iteration routine }
	(B_sxr) (B_szr) (B_sxi) (B_szi)

BOUNDARIES
	region 'space'
	start (srad,0)
	value (A_syr) = 0
	value (A_syi) = 0
	arc (center = 0,0)	angle = 360

	region 'um'
	sigma = sigma_um
	mesh_spacing = mrad/dens
	start(0,mrad)
	natural(A_syr) = mu0*tau*E_r		{ for ocean BC at Earth-Space interface }
	natural(A_syi) = mu0*tau*E_i 		{ remove these lines for no ocean }
	arc (center = 0,0) angle = 360
	
	region 'tz'
	sigma = sigma_tz
	mesh_spacing = mrad/dens
	start(0,tzt)
	arc (center = 0,0) angle = 360

	region 'lm'
	sigma = sigma_lm
	mesh_spacing = mrad/dens
	start(0,tzb)
	arc (center = 0,0) angle = 360

	region 'core'
	sigma = sigma_core
	start(0,crad)
	arc (center = 0,0) angle = 360
	
MONITORS
	grid (x,z)
	grid (x,z) zoom (-zm,-zm,2*zm,2*zm)
	contour (sigma/M^3) painted zoom (-zm,-zm,2*zm,2*zm)
	contour (E_r) zoom (-zm,-zm,2*zm,2*zm) painted
	contour (E_i) zoom (-zm,-zm,2*zm,2*zm) painted
	vector (B_pr) zoom (-zm,-zm,2*zm,2*zm) fixed range (0,4.2e-8)
	vector (B_sr) zoom (-zm,-zm,2*zm,2*zm) fixed range (0,5e-8)
	vector (B_si) zoom (-zm,-zm,2*zm,2*zm) fixed range (0,4e-8)
	vector (B_tr) zoom (-zm,-zm,2*zm,2*zm) fixed range (0,6e-8)
	elevation(A_tyr,A_tyi,A_syr,A_syi,A_pyr,A_pyi) from (-2*mrad,0) to (2*mrad,0)
	elevation(B_szr,B_szi,B_tzr,B_pzr) from (-2*mrad,0) to (2*mrad,0)
	
PLOTS
	grid (x,z)
	grid (x,z) zoom (-zm,-zm,2*zm,2*zm)
	contour (sigma/M^3) painted zoom (-zm,-zm,2*zm,2*zm)
	contour (E_r) zoom (-zm,-zm,2*zm,2*zm) painted
	contour (E_i) zoom (-zm,-zm,2*zm,2*zm) painted
	vector (B_pr) zoom (-zm,-zm,2*zm,2*zm) fixed range (0,4.2e-8)
	vector (B_sr) zoom (-zm,-zm,2*zm,2*zm) fixed range (0,5e-8)
	vector (B_si) zoom (-zm,-zm,2*zm,2*zm) fixed range (0,4e-8)
	vector (B_tr) zoom (-zm,-zm,2*zm,2*zm) fixed range (0,6e-8)
	elevation(A_tyr,A_tyi,A_syr,A_syi,A_pyr,A_pyi) from (-2*mrad,0) to (2*mrad,0)
	elevation(B_szr,B_szi,B_tzr,B_pzr) from (-2*mrad,0) to (2*mrad,0)

	{ export values for satellite-track c estimate }
	SUMMARY EXPORT noheader file="track.txt" merge
	{ long=0 => x-axis, long=180 => -x-axis }
	report(logperiod)
	repeat colat = 40 by .5 to 140		{ 201 colatitudes }
		report(colat)
		report(val(B_txr, sat*sin(colat*pi/180),sat*cos(colat*pi/180)))
		report(val(B_tzr, sat*sin(colat*pi/180),sat*cos(colat*pi/180)))
		report(val(B_txi, sat*sin(colat*pi/180),sat*cos(colat*pi/180)))
		report(val(B_tzi, sat*sin(colat*pi/180),sat*cos(colat*pi/180)))
	endrepeat
	
END

