{ This script is frequency-domain induction in a concentric 3D Earth with spatially 
uniform, purely real, periodic, external B-field in the z-direction only, using 
the magnetic vector potential.  x is to the right, y is into the screen, z is up.
This is a limited 3D problem, so V and A_z vanish.  To translate for a fully 3D 
problem, include these variables in the code.
Note on units: length units are Mm = 1e6m.  Thus we must convert permeability (1H/m =1 kg*m/(s*A)^ 2),
and conductivity (1 S/m = 1s^ 3*A^ 2/(kg*m^ 3)).}

TITLE '3D induction using vector potential in the frequency domain'


SELECT
	ngrid = 2				{ for large cells at space boundary }
	stages = 35				{ number of frequencies to simulate }
	threads = 4				{ number of CPU cores to use }
	prefer_speed			
	errlim = 1e-3				{ error tolerance }

COORDINATES
	cartesian3

VARIABLES
	{ For limited 3D problem }
 	A_sr = Vector(A_sxr,A_syr,0) 		{ magnetic vector potential }
 	A_si = Vector(A_sxi,A_syi,0) 									

	{ For full 3D problem }
{ 	A_sr = Vector(A_sxr,A_syr,A_szr) 	{ magnetic vector potential }
 	A_si = Vector(A_sxi,A_syi,A_szi) 									
	V_r					{ electric scalar potential }
	V_i					
}
DEFINITIONS
	dens = 11				{ for determining mesh density }
	weight = 0
	M = 1e6					{ 1 million }
	mu0 = 4*pi*1e-7/M			{ permeability of free space (see note above) }	
	mrad = 6.371				{ radius of mantle in Mm }
	srad = 10*mrad				{ full radius of modeled space }
	crad = 3.5				{ radius of core}
	tzb = mrad - .8				{ radius of conductivity interfaces }
	tzt = mrad - .4	
	zm = 1.5*mrad				{ for zooming plots } 	
	sat_alt = .4	
	sat = mrad + sat_alt
	B0 = 100e-9				{ amplitude of primary field }	
	logperiod = -4/24*(stage)+8+4/24	{ goes from a few minutes to ~3 years in 25 steps }
	period = 10^logperiod			{ primary field period in seconds }
	f = 1/period
	w = 2*pi*f				{ angular frequency of A_e }
	sigma = 0				{ space conductivity }
	sigma1 = sigma/M^3
	sigma_core = 5e5*M^3			{ core conductivity }
	sigma_lm = 1*M^3			{ lower mantle conductivity }
	sigma_tz = .1*M^3			{ transition zone conductivity }
	sigma_um = .01*M^3			{ upper mantle conductivity }
{	tau = 9000*M^2				{ ocean conductance }
}	switch = if sigma<>0 then 0 else 1	{ for switching V to laplace equation in space }

	{ create primary (external) potential and field (downward-pointing)}
	A_pxr = B0/2*y				{ Re(x-comp) of primary vector potential }
	A_pyr = -B0/2*x				{ Re(y-comp) of primary vector potential }
	A_pzr = 0				{ Re(z-comp) of primary vector potential }
	A_pr = vector(A_pxr,A_pyr,A_pzr)	
	B_pr = curl(A_pr)			{ Re(primary mag field) }
	B_pxr = xcomp(B_pr)
	B_pyr = ycomp(B_pr)
	B_pzr = zcomp(B_pr)

	{ create secondary (internal) potential and field }
	{ Remove the next line for full 3D model }
	A_szr = 0
	{ Remove the next line for full 3D model }
	A_szi = 0
	B_sr = curl(A_sr)			{ Re(secondary mag field) }
	B_sxr = xcomp(B_sr)
	B_syr = ycomp(B_sr)
	B_szr = zcomp(B_sr)
	B_si = curl(A_si)			{ Im(secondary mag field) }
	B_sxi = xcomp(B_si)
	B_syi = ycomp(B_si)
	B_szi = zcomp(B_si)

	{ create total potential and field }
	A_txr = A_pxr + A_sxr			{ Re(x-comp) of total vector pot'l }
	A_tyr = A_pyr + A_syr			{ Re(y-comp) of total vector pot'l }
	A_tzr = A_pzr + A_szr			{ Re(z-comp) of total vector pot'l }
	A_tr = vector(A_txr,A_tyr,A_tzr)	
	A_txi = A_sxi				{ Im(x-comp) of total vector pot'l }
	A_tyi = A_syi				{ Im(y-comp) of total vector pot'l }
	A_tzi = A_szi				{ Im(z-comp) of total vector pot'l }
	A_ti = vector(A_txi,A_tyi,A_tzi)
	B_txr = B_pxr + B_sxr
	B_tyr = B_pyr + B_syr
	B_tzr = B_pzr + B_szr
	B_tr = vector(B_txr,B_tyr,B_tzr)	{ Re(total mag field) }
	B_ti = B_si				{ Im(total mag field) }
	B_txi = B_sxi
	B_tyi = B_syi
	B_tzi = B_szi
	E_r = w*A_ti{-grad(V_r)}		{ Re(induced electric field) }
	E_xr = xcomp(E_r)
	E_yr = ycomp(E_r)
	E_zr = zcomp(E_r)
	E_i = -w*A_tr{-grad(V_i)}		{ Im(induced electric field) }
	E_xi = xcomp(E_i)
	E_yi = ycomp(E_i)
	E_zi = zcomp(E_i)


EQUATIONS
	A_sr:	del2(A_sr) = -w*mu0*sigma*A_ti
	A_si: 	del2(A_si)  = w*mu0*sigma*A_tr
{	A_sr:	del2(A_sr)  = -w*mu0*sigma*A_ti + mu0*sigma*grad(V_r)
	A_si: 	del2(A_si)  = w*mu0*sigma*A_tr + mu0*sigma*grad(V_i)
	V_r: 	div(sigma*(grad(V_r) - w*A_ti)) + switch*del2(V_r) = 0
	V_i: 	div(sigma*(grad(V_i) + w*A_tr)) + switch*del2(V_i) = 0
}

RESOLVE 		{ Include mag fields in error estimate for iteration routine }
	(B_sxr),(B_sxi),(B_syr),(B_syi),(B_szr),(B_szi)

EXTRUSION
	surface 'space_bottom'  z = -sphere ((0,0,0),srad)
		layer 'space_under'
	surface 'earth_bottom'  z = -sphere ((0,0,0),mrad)
		layer 'um_under'
	surface 'tzt_bottom'  z = -sphere ((0,0,0),tzt)
		layer 'tz_under'
	surface 'tzb_bottom'  z = -sphere ((0,0,0),tzb)
		layer 'lm_under'
	surface 'core_bottom'  z = -sphere ((0,0,0),crad)
		layer 'core_layer'
	surface 'core_top'  z = sphere ((0,0,0),crad)
		layer 'lm_over'
	surface 'tzb_top'  z = sphere ((0,0,0),tzb)
		layer 'tz_over'
	surface 'tzt_top'  z = sphere ((0,0,0),tzt)
		layer 'um_over'
	surface 'earth_top'  z = sphere ((0,0,0),mrad)
		layer 'space_over'
	surface 'space_top'  z = sphere ((0,0,0),srad)

BOUNDARIES
	surface 'space_bottom'
		mesh_spacing = srad/2
	  	value (A_sxr) = 0 	
	  	value (A_syr) = 0 	
	  	value (A_sxi) = 0 	
	  	value (A_syi) = 0 	
{	  	value (A_szr) = 0 	
	  	value (A_szi) = 0 	
		value(V_r) = 0
		value(V_i) = 0
}
	surface 'space_top'
		mesh_spacing = srad/2
	  	value (A_sxr) = 0 	
	  	value (A_syr) = 0 	
	  	value (A_sxi) = 0 	
	  	value (A_syi) = 0 	
{	  	value (A_szr) = 0 	
	  	value (A_szi) = 0 	
		value(V_r) = 0
		value(V_i) = 0
}
	region 1 'space'
		layer 'space_under' 
			mesh_spacing = srad/2
		layer 'space_over' 
			mesh_spacing = srad/2
		start (srad,0)
	  	value (A_sxr) = 0 	
	  	value (A_syr) = 0 	
	  	value (A_sxi) = 0 	
	  	value (A_syi) = 0 	
{	  	value (A_szr) = 0 	
	  	value (A_szi) = 0 	
		value(V_r) = 0
		value(V_i) = 0
}		arc(center=0,0) angle=360 to close

	limited region 2  'upper_mantle'
	surface'earth_bottom'
		mesh_spacing = mrad/dens
		{ For including uniform global surface layer }
{		natural(A_sxr) = mu0*tau*E_xr
		natural(A_sxi) = mu0*tau*E_xi
		natural(A_syr) = mu0*tau*E_yr
		natural(A_syi) = mu0*tau*E_yi
{		natural(A_szr) = mu0*tau*E_zr
		natural(A_szi) = mu0*tau*E_zi
}}
	surface'earth_top'
		mesh_spacing = mrad/dens
		{ For including uniform global surface layer }
{		natural(A_sxr) = mu0*tau*E_xr
		natural(A_sxi) = mu0*tau*E_xi
		natural(A_syr) = mu0*tau*E_yr
		natural(A_syi) = mu0*tau*E_yi
{		natural(A_szr) = mu0*tau*E_zr
		natural(A_szi) = mu0*tau*E_zi
}}	
		layer 'um_under'
			sigma = sigma_um
			mesh_spacing = mrad/dens
		layer 'um_over'
			sigma = sigma_um
			mesh_spacing = mrad/dens
		layer 'tz_under' layer 'lm_under' layer 'core_layer'
		layer 'tz_over' layer 'lm_over'
		start (mrad,0)
		{ For including uniform global surface layer }
		natural(A_sxr) = mu0*tau*E_xr
{		natural(A_sxi) = mu0*tau*E_xi
		natural(A_syr) = mu0*tau*E_yr
		natural(A_syi) = mu0*tau*E_yi
{		natural(A_szr) = mu0*tau*E_zr
		natural(A_szi) = mu0*tau*E_zi
}}		arc(center=0,0) angle=360

	limited region 4  'transition_zone'
		layer 'tz_under'
			sigma = sigma_tz
			mesh_spacing = mrad/dens
		layer 'tz_over'
			sigma = sigma_tz
			mesh_spacing = mrad/dens
		layer 'lm_under' layer 'core_layer'
		layer 'lm_over'
		start (tzt,0)
		arc(center=0,0) angle=360

	limited region 5  'lower_mantle'
		layer 'lm_under'
			sigma = sigma_lm
			mesh_spacing = mrad/dens
		layer 'lm_over'
			sigma = sigma_lm
			mesh_spacing = mrad/dens
			layer 'core_layer'
		start (tzb,0)
		arc(center=0,0) angle=360

	limited region 6  'core'
		layer 'core_layer'
			sigma = sigma_core
		start (crad,0)
		arc(center=0,0) angle=360

MONITORS   
	grid(x,z) on y=0 zoom (-zm,-zm,2*zm,2*zm)
	grid(x,y) on z=0 zoom (-zm,-zm,2*zm,2*zm)
	contour (sigma/M^3) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_pyr) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_syr) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_syi) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_tyr) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_pxr) painted zoom (-zm,-zm,2*zm,2*zm) on x = 0
	contour (A_sxr) painted zoom (-zm,-zm,2*zm,2*zm) on x = 0
	contour (A_sxi) painted zoom (-zm,-zm,2*zm,2*zm) on x= 0
	contour (A_txr) painted zoom (-zm,-zm,2*zm,2*zm) on x = 0
	vector (A_pxr,A_pyr) zoom (-zm,-zm,2*zm,2*zm) on z = 0
	vector (A_sxr,A_syr) zoom (-zm,-zm,2*zm,2*zm) on z = 0	
	vector (A_sxi,A_syi) zoom (-zm,-zm,2*zm,2*zm) on z = 0	
	vector (A_txr,A_tyr) zoom (-zm,-zm,2*zm,2*zm) on z = 0	
	vector (B_pxr,B_pzr) zoom (-zm,-zm,2*zm,2*zm) on y = 0 fixed range (0,1.3*B0)
	vector (B_sxr,B_szr) zoom (-zm,-zm,2*zm,2*zm) on y = 0 fixed range (0,B0)
	vector (B_txr,B_tzr) zoom (-zm,-zm,2*zm,2*zm) on y = 0 fixed range (0,2*B0)
	vector (B_txi,B_tzi) zoom (-zm,-zm,2*zm,2*zm) on y = 0 fixed range (0,B0)
	vector (B_pyr,B_pzr) zoom (-zm,-zm,2*zm,2*zm) on x = 0 fixed range (0,1.3*B0)
	vector (B_syr,B_szr) zoom (-zm,-zm,2*zm,2*zm) on x = 0 fixed range (0,B0)
	vector (B_tyr,B_tzr) zoom (-zm,-zm,2*zm,2*zm) on x = 0 fixed range (0,2*B0)
	vector (B_tyi,B_tzi) zoom (-zm,-zm,2*zm,2*zm) on x = 0 fixed range (0,B0)

PLOTS
	grid(x,z) on y=0 zoom (-zm,-zm,2*zm,2*zm)
	grid(x,y) on z=0 zoom (-zm,-zm,2*zm,2*zm)
	contour (sigma/M^3) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_pyr) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_syr) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_syi) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_tyr) painted zoom (-zm,-zm,2*zm,2*zm) on y = 0
	contour (A_pxr) painted zoom (-zm,-zm,2*zm,2*zm) on x = 0
	contour (A_sxr) painted zoom (-zm,-zm,2*zm,2*zm) on x = 0
	contour (A_sxi) painted zoom (-zm,-zm,2*zm,2*zm) on x= 0
	contour (A_txr) painted zoom (-zm,-zm,2*zm,2*zm) on x = 0
	vector (A_pxr,A_pyr) zoom (-zm,-zm,2*zm,2*zm) on z = 0
	vector (A_sxr,A_syr) zoom (-zm,-zm,2*zm,2*zm) on z = 0	
	vector (A_sxi,A_syi) zoom (-zm,-zm,2*zm,2*zm) on z = 0	
	vector (A_txr,A_tyr) zoom (-zm,-zm,2*zm,2*zm) on z = 0	
	vector (B_pxr,B_pzr) zoom (-zm,-zm,2*zm,2*zm) on y = 0 fixed range (0,1.3*B0)
	vector (B_sxr,B_szr) zoom (-zm,-zm,2*zm,2*zm) on y = 0 fixed range (0,B0)
	vector (B_txr,B_tzr) zoom (-zm,-zm,2*zm,2*zm) on y = 0 fixed range (0,2*B0)
	vector (B_txi,B_tzi) zoom (-zm,-zm,2*zm,2*zm) on y = 0 fixed range (0,B0)
	vector (B_pyr,B_pzr) zoom (-zm,-zm,2*zm,2*zm) on x = 0 fixed range (0,1.3*B0)
	vector (B_syr,B_szr) zoom (-zm,-zm,2*zm,2*zm) on x = 0 fixed range (0,B0)
	vector (B_tyr,B_tzr) zoom (-zm,-zm,2*zm,2*zm) on x = 0 fixed range (0,2*B0)
	vector (B_tyi,B_tzi) zoom (-zm,-zm,2*zm,2*zm) on x = 0 fixed range (0,B0)
	contour (error) painted zoom (-zm,-zm,2*zm,2*zm) on x = 0
	contour (error) painted zoom (-zm,-zm,2*zm,2*zm) on z = 0


	{ export values for satellite-track c estimate }
	SUMMARY EXPORT noheader file="B_track.txt" merge
	{ long=0 => x-axis, long=180 => -x-axis, long=90 => y-axis, long=270 => -y-axis }
	report(logperiod)
	repeat long = 0 by 90 to 270
		repeat colat = 40 by .5 to 140		{ 201 colatitudes }
			report(colat)
			report(long)
			report(val(B_txr, sat*cos(long*pi/180)*sin(colat*pi/180),sat*sin(long*pi/180)*sin(colat*pi/180),sat*cos(colat*pi/180)))
			report(val(B_tyr, sat*cos(long*pi/180)*sin(colat*pi/180),sat*sin(long*pi/180)*sin(colat*pi/180),sat*cos(colat*pi/180)))
			report(val(B_tzr, sat*cos(long*pi/180)*sin(colat*pi/180),sat*sin(long*pi/180)*sin(colat*pi/180),sat*cos(colat*pi/180)))
			report(val(B_txi, sat*cos(long*pi/180)*sin(colat*pi/180),sat*sin(long*pi/180)*sin(colat*pi/180),sat*cos(colat*pi/180)))
			report(val(B_tyi, sat*cos(long*pi/180)*sin(colat*pi/180),sat*sin(long*pi/180)*sin(colat*pi/180),sat*cos(colat*pi/180)))
			report(val(B_tzi, sat*cos(long*pi/180)*sin(colat*pi/180),sat*sin(long*pi/180)*sin(colat*pi/180),sat*cos(colat*pi/180)))
		endrepeat
	endrepeat

	
END
